const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');

const srcDir = path.join(__dirname, 'src');
const outDir = path.join(__dirname, 'out');

if (!fs.existsSync(outDir)) {
    fs.mkdirSync(outDir, { recursive: true });
}

const compileCss = (filename) => {
    const cssSrc = path.join(srcDir, filename);
    const cssDest = path.join(outDir, `${path.parse(filename).name}.min.css`);

    // Vérifie si le chemin source est un fichier CSS
    if (fs.existsSync(cssSrc) && fs.statSync(cssSrc).isFile()) {
        // Minifier le CSS
        const minifiedCss = execSync(`uglifycss ${cssSrc}`).toString();

        // Écrire le CSS minifié dans le fichier de destination
        fs.writeFileSync(cssDest, minifiedCss);

        console.log(`Compilation et minification du fichier CSS ${filename} terminées.`);
    } else {
        console.error(`Le fichier CSS ${filename} n'existe pas ou n'est pas un fichier.`);
    }
};

const compileAllCss = () => {
    const files = fs.readdirSync(srcDir).filter(file => path.extname(file) === '.css');

    files.forEach(file => {
        compileCss(file);
    });
};

const compileThemeCss = (theme, outputFilename) => {
    const themeCssSrc = path.join(srcDir, theme + '.css');
    let cssDest;

    if (outputFilename) {
        cssDest = path.join(outDir, `${outputFilename}.min.css`);
    } else {
        cssDest = path.join(outDir, `${theme}-min.css`);
    }

    // Vérifie si le chemin source est un fichier CSS
    if (fs.existsSync(themeCssSrc) && fs.statSync(themeCssSrc).isFile()) {
        // Minifier le CSS
        const minifiedCss = execSync(`uglifycss ${themeCssSrc}`).toString();

        // Écrire le CSS minifié dans le fichier de destination
        fs.writeFileSync(cssDest, minifiedCss);

        console.log(`Compilation et minification du thème CSS ${theme} terminées.`);
    } else {
        console.error(`Le fichier CSS ${themeCssSrc} n'existe pas ou n'est pas un fichier.`);
    }
};

// Récupère les arguments de ligne de commande
const args = process.argv.slice(2);

// Vérifie les arguments de ligne de commande pour décider quelle méthode exécuter
if (args.length === 0 || args[0] === '--all') {
    // Aucun argument ou 'all' passé en tant qu'argument, exécute la méthode pour tous les fichiers CSS disponibles
    compileAllCss();
} else if (args[0] === '--theme') {
    // 'theme' passé en tant qu'argument, exécute la méthode pour un thème choisi
    const themeToCompile = args[1]; // Le nom du thème choisi doit être passé en tant que deuxième argument
    const outputFilename = args[2]; // Le nom du fichier de sortie peut être passé en tant que troisième argument (optionnel)
    compileThemeCss(themeToCompile, outputFilename);
} else {
    console.error('Usage: node build-css.js [--all | --theme <theme_name> [<output_filename>]]');
    console.error('--all: Compile et minifie tous les fichiers CSS du répertoire source vers le répertoire de sortie');
    console.error('--theme <theme_name> [<output_filename>]: Compile et minifie le CSS pour un thème spécifique');
}
